package jadean.dean.java.resourceparser.javaparser.visitors;
import java.util.Stack;

import japa.parser.ast.BlockComment;
import japa.parser.ast.CompilationUnit;
import japa.parser.ast.ImportDeclaration;
import japa.parser.ast.LineComment;
import japa.parser.ast.PackageDeclaration;
import japa.parser.ast.TypeParameter;
import japa.parser.ast.body.AnnotationDeclaration;
import japa.parser.ast.body.AnnotationMemberDeclaration;
import japa.parser.ast.body.BodyDeclaration;
import japa.parser.ast.body.ClassOrInterfaceDeclaration;
import japa.parser.ast.body.ConstructorDeclaration;
import japa.parser.ast.body.EmptyMemberDeclaration;
import japa.parser.ast.body.EmptyTypeDeclaration;
import japa.parser.ast.body.EnumConstantDeclaration;
import japa.parser.ast.body.EnumDeclaration;
import japa.parser.ast.body.FieldDeclaration;
import japa.parser.ast.body.InitializerDeclaration;
import japa.parser.ast.body.JavadocComment;
import japa.parser.ast.body.MethodDeclaration;
import japa.parser.ast.body.Parameter;
import japa.parser.ast.body.TypeDeclaration;
import japa.parser.ast.body.VariableDeclarator;
import japa.parser.ast.body.VariableDeclaratorId;
import japa.parser.ast.expr.AnnotationExpr;
import japa.parser.ast.expr.ArrayAccessExpr;
import japa.parser.ast.expr.ArrayCreationExpr;
import japa.parser.ast.expr.ArrayInitializerExpr;
import japa.parser.ast.expr.AssignExpr;
import japa.parser.ast.expr.BinaryExpr;
import japa.parser.ast.expr.BooleanLiteralExpr;
import japa.parser.ast.expr.CastExpr;
import japa.parser.ast.expr.CharLiteralExpr;
import japa.parser.ast.expr.ClassExpr;
import japa.parser.ast.expr.ConditionalExpr;
import japa.parser.ast.expr.DoubleLiteralExpr;
import japa.parser.ast.expr.EnclosedExpr;
import japa.parser.ast.expr.Expression;
import japa.parser.ast.expr.FieldAccessExpr;
import japa.parser.ast.expr.InstanceOfExpr;
import japa.parser.ast.expr.IntegerLiteralExpr;
import japa.parser.ast.expr.IntegerLiteralMinValueExpr;
import japa.parser.ast.expr.LongLiteralExpr;
import japa.parser.ast.expr.LongLiteralMinValueExpr;
import japa.parser.ast.expr.MarkerAnnotationExpr;
import japa.parser.ast.expr.MemberValuePair;
import japa.parser.ast.expr.MethodCallExpr;
import japa.parser.ast.expr.NameExpr;
import japa.parser.ast.expr.NormalAnnotationExpr;
import japa.parser.ast.expr.NullLiteralExpr;
import japa.parser.ast.expr.ObjectCreationExpr;
import japa.parser.ast.expr.QualifiedNameExpr;
import japa.parser.ast.expr.SingleMemberAnnotationExpr;
import japa.parser.ast.expr.StringLiteralExpr;
import japa.parser.ast.expr.SuperExpr;
import japa.parser.ast.expr.ThisExpr;
import japa.parser.ast.expr.UnaryExpr;
import japa.parser.ast.expr.VariableDeclarationExpr;
import japa.parser.ast.stmt.AssertStmt;
import japa.parser.ast.stmt.BlockStmt;
import japa.parser.ast.stmt.BreakStmt;
import japa.parser.ast.stmt.CatchClause;
import japa.parser.ast.stmt.ContinueStmt;
import japa.parser.ast.stmt.DoStmt;
import japa.parser.ast.stmt.EmptyStmt;
import japa.parser.ast.stmt.ExplicitConstructorInvocationStmt;
import japa.parser.ast.stmt.ExpressionStmt;
import japa.parser.ast.stmt.ForStmt;
import japa.parser.ast.stmt.ForeachStmt;
import japa.parser.ast.stmt.IfStmt;
import japa.parser.ast.stmt.LabeledStmt;
import japa.parser.ast.stmt.ReturnStmt;
import japa.parser.ast.stmt.Statement;
import japa.parser.ast.stmt.SwitchEntryStmt;
import japa.parser.ast.stmt.SwitchStmt;
import japa.parser.ast.stmt.SynchronizedStmt;
import japa.parser.ast.stmt.ThrowStmt;
import japa.parser.ast.stmt.TryStmt;
import japa.parser.ast.stmt.TypeDeclarationStmt;
import japa.parser.ast.stmt.WhileStmt;
import japa.parser.ast.type.ClassOrInterfaceType;
import japa.parser.ast.type.PrimitiveType;
import japa.parser.ast.type.ReferenceType;
import japa.parser.ast.type.Type;
import japa.parser.ast.type.VoidType;
import japa.parser.ast.type.WildcardType;
import japa.parser.ast.visitor.VoidVisitor;

public class PrintXMLVisitor<A> implements VoidVisitor<A> {

	private StringBuilder sb;
	private int indent = 0;
	private Stack<String> tags;
	
	public String getSource() {
		return sb.toString();
	}

	public PrintXMLVisitor() {
		this.sb = new StringBuilder();
		this.indent = 0;
		this.tags = new Stack<String>();
	}

	private void indent() {
		this.indent++;
	}

	private void unindent() {
		this.indent--;
	}

	private void appendSource(String s) {
		for (int i = 0; i < this.indent; i++) {
			this.sb.append(" ");
		}
		this.sb.append(s + "\n");
	}

	private void openTag(String s) {
		this.appendSource("<" + s + ">");
		this.indent();
		this.tags.push(s);
	}

	private void closeTag() {
		this.unindent();
		String s = tags.pop();
		this.appendSource("</" + s + ">");
	}

	public void visit(AnnotationDeclaration n, A arg) {
		this.openTag("AnnotationDeclaration");
		if (n.getJavaDoc() != null) {
			n.getJavaDoc().accept(this, arg);
		}
		if (n.getAnnotations() != null) {
			this.openTag("Annotations");
			for (AnnotationExpr a : n.getAnnotations()) {
				a.accept(this, arg);
			}
			this.closeTag();
		}
		if (n.getMembers() != null) {
			this.openTag("Members");
			for (BodyDeclaration member : n.getMembers()) {
				member.accept(this, arg);
			}
			this.closeTag();
		}
		this.closeTag();
	}

	public void visit(AnnotationMemberDeclaration n, A arg) {
		this.openTag("AnnotationMemberDeclaration");
		if (n.getJavaDoc() != null) {
			n.getJavaDoc().accept(this, arg);
		}
		if (n.getAnnotations() != null) {
			this.openTag("Annotations");
			for (AnnotationExpr a : n.getAnnotations()) {
				a.accept(this, arg);
			}
			this.closeTag();
		}
		this.openTag("Type");
		n.getType().accept(this, arg);
		this.closeTag();
		if (n.getDefaultValue() != null) {
			this.openTag("DefaultValue");
			n.getDefaultValue().accept(this, arg);
			this.closeTag();
		}
		this.closeTag();
	}

	public void visit(ArrayAccessExpr n, A arg) {
		this.openTag("ArrayAccessExpr");
		this.openTag("Name");
		n.getName().accept(this, arg);
		this.closeTag();
		this.openTag("Index");
		n.getIndex().accept(this, arg);
		this.closeTag();
		this.closeTag();
	}

	public void visit(ArrayCreationExpr n, A arg) {
		this.openTag("ArrayCreationExpr");
		this.openTag("Type");
		n.getType().accept(this, arg);
		this.closeTag();
		if (n.getDimensions() != null) {
			this.openTag("Dimensions");
			for (Expression dim : n.getDimensions()) {
				dim.accept(this, arg);
			}
			this.closeTag();
		} else {
			this.openTag("Initializer");
			n.getInitializer().accept(this, arg);
			this.closeTag();
		}
		this.closeTag();        
	}

	public void visit(ArrayInitializerExpr n, A arg) {
		this.openTag("ArrayInitializerExpr");
		if (n.getValues() != null) {
			this.openTag("Values");
			for (Expression expr : n.getValues()) {
				expr.accept(this, arg);
			}
			this.closeTag();
		}
		this.closeTag();        
	}

	public void visit(AssertStmt n, A arg) {
		this.openTag("AssertStmt");
		this.openTag("Check");
		n.getCheck().accept(this, arg);
		this.closeTag();
		if (n.getMessage() != null) {
			this.openTag("Message");
			n.getMessage().accept(this, arg);
			this.closeTag();
		}
		this.closeTag();            
	}

	public void visit(AssignExpr n, A arg) {
		this.openTag("AssignExpr");
		this.openTag("Target");
		n.getTarget().accept(this, arg);
		this.closeTag();
		this.openTag("Value");
		n.getValue().accept(this, arg);
		this.closeTag();
		this.closeTag();
	}

	public void visit(BinaryExpr n, A arg) {
		this.openTag("BinaryExpr");
		this.openTag("Left");
		n.getLeft().accept(this, arg);
		this.closeTag();  
		this.openTag("Operator");
		this.appendSource(n.getOperator().name());
		this.closeTag();    
		this.openTag("Right");
		n.getRight().accept(this, arg);
		this.closeTag();
		this.closeTag();
	}

	public void visit(BlockComment n, A arg) {
	}

	public void visit(BlockStmt n, A arg) {
		this.openTag("BlockStmt");
		if (n.getStmts() != null) {
			this.openTag("Statements");
			for (Statement s : n.getStmts()) {
				s.accept(this, arg);
			}
			this.closeTag();
		}
		this.closeTag();
	}

	public void visit(BooleanLiteralExpr n, A arg) {
		this.openTag("BooleanLiteralExpr");
		this.appendSource(n.getValue() + "");
		this.closeTag();   	
	}

	public void visit(BreakStmt n, A arg) {
		this.openTag("BreakStmt");
		this.closeTag();      	
	}

	public void visit(CastExpr n, A arg) {
		this.openTag("CastExpr");
		this.openTag("Type");
		n.getType().accept(this, arg);
		this.closeTag();
		this.openTag("Expression");
		n.getExpr().accept(this, arg);
		this.closeTag();
		this.closeTag();       
	}

	public void visit(CatchClause n, A arg) {
		this.openTag("CatchClause");
		this.openTag("Except");
		n.getExcept().accept(this, arg);
		this.closeTag();
		this.openTag("Catch");
		n.getCatchBlock().accept(this, arg);
		this.closeTag();
		this.closeTag();      
	}

	public void visit(CharLiteralExpr n, A arg) {
		this.openTag("CharLiteralExpr");
		this.appendSource(n.getValue());
		this.closeTag();
	}

	public void visit(ClassExpr n, A arg) {
		this.openTag("ClassExpr");
		n.getType().accept(this, arg);
		this.closeTag();         
	}

	public void visit(ClassOrInterfaceDeclaration n, A arg) {
		this.openTag("ClassOrInterfaceDeclaration");
		this.openTag("Name");
		this.appendSource(n.getName());
		this.closeTag();    	
		if (n.getJavaDoc() != null) {
			n.getJavaDoc().accept(this, arg);
		}
		if (n.getAnnotations() != null) {
			this.openTag("Annotations");
			for (AnnotationExpr a : n.getAnnotations()) {
				a.accept(this, arg);
			}
			this.closeTag();
		}

		if (n.getTypeParameters() != null) {
			this.openTag("TypeParameters");
			for (TypeParameter t : n.getTypeParameters()) {
				t.accept(this, arg);
			}
			this.closeTag();
		}

		if (n.getExtends() != null) {
			this.openTag("Extends");
			for (ClassOrInterfaceType c : n.getExtends()) {
				c.accept(this, arg);
			}
			this.closeTag();
		}

		if (n.getImplements() != null) {
			this.openTag("Implements");
			for (ClassOrInterfaceType c : n.getImplements()) {
				c.accept(this, arg);
			}
			this.closeTag();
		}

		if (n.getMembers() != null) {
			this.openTag("Members");
			for (BodyDeclaration member : n.getMembers()) {
				this.openTag("BodyDeclaration");
				member.accept(this, arg);
				this.closeTag();
			}
			this.closeTag();
		}
		this.closeTag();
	}

	public void visit(ClassOrInterfaceType n, A arg) {
		this.openTag("ClassOrInterfaceType");
		this.openTag("Name");
		this.appendSource(n.getName());
		this.closeTag();
		if (n.getScope() != null) {
			this.openTag("Scope");
			n.getScope().accept(this, arg);
			this.closeTag();
		}
		if (n.getTypeArgs() != null) {
			this.openTag("TypeArguments");
			for (Type t : n.getTypeArgs()) {
				t.accept(this, arg);
			}
			this.closeTag();
		}
		this.closeTag();
	}

	public void visit(CompilationUnit n, A arg) {
		this.openTag("CompilationUnit");
		if (n.getPackage() != null) {
			n.getPackage().accept(this, arg);
		}
		if (n.getImports() != null) {
			this.openTag("Imports");
			for (ImportDeclaration i : n.getImports()) {
				i.accept(this, arg);
			}
			this.closeTag();
		}
		if (n.getTypes() != null) {
			this.openTag("Types");
			for (TypeDeclaration typeDeclaration : n.getTypes()) {
				typeDeclaration.accept(this, arg);
			}
			this.closeTag();
		}
		this.closeTag();
	}

	public void visit(ConditionalExpr n, A arg) {
		this.openTag("ConditionalExpr");
		this.openTag("Condition");
		n.getCondition().accept(this, arg);
		this.closeTag();
		this.openTag("ThenExpression");
		n.getThenExpr().accept(this, arg);
		this.closeTag();
		this.openTag("ElseExpression");
		n.getElseExpr().accept(this, arg);
		this.closeTag();
		this.closeTag();
	}

	public void visit(ConstructorDeclaration n, A arg) {
		this.openTag("ConstructorDeclaration");
		if (n.getJavaDoc() != null) {
			n.getJavaDoc().accept(this, arg);
		}
		if (n.getAnnotations() != null) {
			this.openTag("Annotations");
			for (AnnotationExpr a : n.getAnnotations()) {
				a.accept(this, arg);
			}
			this.closeTag();
		}
		if (n.getTypeParameters() != null) {
			this.openTag("TypeParameters");
			for (TypeParameter t : n.getTypeParameters()) {
				t.accept(this, arg);
			}
			this.closeTag();
		}
		if (n.getParameters() != null) {
			this.openTag("Parameters");
			for (Parameter p : n.getParameters()) {
				p.accept(this, arg);
			}
			this.closeTag();
		}
		if (n.getThrows() != null) {
			this.openTag("Throws");
			for (NameExpr name : n.getThrows()) {
				name.accept(this, arg);
			}
			this.closeTag();
		}
		this.openTag("Block");
		n.getBlock().accept(this, arg);
		this.closeTag();
		this.closeTag();        
	}

	public void visit(ContinueStmt n, A arg) {
		this.openTag("ContinueStmt");
		this.closeTag();      	
	}

	public void visit(DoStmt n, A arg) {
		this.openTag("DoStmt");
		this.openTag("Body");
		n.getBody().accept(this, arg);
		this.closeTag();
		this.openTag("Condition");
		n.getCondition().accept(this, arg);
		this.closeTag();        
		this.closeTag();         
	}

	public void visit(DoubleLiteralExpr n, A arg) {
		this.openTag("ContinueStmt");
		this.appendSource(n.getValue());
		this.closeTag();
	}

	public void visit(EmptyMemberDeclaration n, A arg) {
		this.openTag("EmptyMemberDeclaration");
		if (n.getJavaDoc() != null) {
			n.getJavaDoc().accept(this, arg);
		}
		this.closeTag();           
	}

	public void visit(EmptyStmt n, A arg) {
		this.openTag("EmptyStmt");
		this.closeTag();        	
	}

	public void visit(EmptyTypeDeclaration n, A arg) {
		this.openTag("EmptyTypeDeclaration");
		if (n.getJavaDoc() != null) {
			n.getJavaDoc().accept(this, arg);
		}
		this.closeTag();         
	}

	public void visit(EnclosedExpr n, A arg) {
		this.openTag("EnclosedExpr");
		n.getInner().accept(this, arg);
		this.closeTag();         
	}

	public void visit(EnumConstantDeclaration n, A arg) {
		this.openTag("EnumConstantDeclaration");
		if (n.getJavaDoc() != null) {
			n.getJavaDoc().accept(this, arg);
		}
		if (n.getAnnotations() != null) {
			this.openTag("Annotations");
			for (AnnotationExpr a : n.getAnnotations()) {
				a.accept(this, arg);
			}
			this.closeTag();
		}
		if (n.getArgs() != null) {
			this.openTag("Arguments");
			for (Expression e : n.getArgs()) {
				e.accept(this, arg);
			}
			this.closeTag();
		}
		if (n.getClassBody() != null) {
			this.openTag("Body");
			for (BodyDeclaration member : n.getClassBody()) {
				member.accept(this, arg);
			}
			this.closeTag();
		}
		this.closeTag();           
	}

	public void visit(EnumDeclaration n, A arg) {
		this.openTag("EnumDeclaration");
		if (n.getJavaDoc() != null) {
			n.getJavaDoc().accept(this, arg);
		}
		if (n.getAnnotations() != null) {
			this.openTag("Annotations");
			for (AnnotationExpr a : n.getAnnotations()) {
				a.accept(this, arg);
			}
			this.closeTag();
		}
		if (n.getImplements() != null) {
			this.openTag("Implements");
			for (ClassOrInterfaceType c : n.getImplements()) {
				c.accept(this, arg);
			}
			this.closeTag();
		}
		if (n.getEntries() != null) {
			this.openTag("Entries");
			for (EnumConstantDeclaration e : n.getEntries()) {
				e.accept(this, arg);
			}
			this.closeTag();
		}
		if (n.getMembers() != null) {
			this.openTag("Members");
			for (BodyDeclaration member : n.getMembers()) {
				member.accept(this, arg);
			}
			this.closeTag();
		}
		this.closeTag();       
	}

	public void visit(ExplicitConstructorInvocationStmt n, A arg) {
		this.openTag("ExplicitConstructorInvocationStmt");
		if (!n.isThis()) {
			if (n.getExpr() != null) {
				this.openTag("Expression");
				n.getExpr().accept(this, arg);
				this.closeTag();
			}
		}
		if (n.getTypeArgs() != null) {
			this.openTag("TypeArguments");
			for (Type t : n.getTypeArgs()) {
				t.accept(this, arg);
			}
			this.closeTag();
		}
		if (n.getArgs() != null) {
			this.openTag("Arguments");
			for (Expression e : n.getArgs()) {
				e.accept(this, arg);
			}
			this.closeTag();
		}
		this.closeTag();
	}

	public void visit(ExpressionStmt n, A arg) {
		this.openTag("ExpressionStmt");
		n.getExpression().accept(this, arg);
		this.closeTag();
	}

	public void visit(FieldAccessExpr n, A arg) {
		this.openTag("FieldAccessExpr");
		this.openTag("FieldField");
		this.appendSource(n.getField());
		this.closeTag();
		this.openTag("Scope");
		n.getScope().accept(this, arg);
		this.closeTag();
		this.closeTag();        
	}

	public void visit(FieldDeclaration n, A arg) {
		this.openTag("FieldDeclaration");
		if (n.getJavaDoc() != null) {
			n.getJavaDoc().accept(this, arg);
		}
		if (n.getAnnotations() != null) {
			this.openTag("Annotations");
			for (AnnotationExpr a : n.getAnnotations()) {
				a.accept(this, arg);
			}
			this.closeTag();
		}
		this.openTag("Type");
		n.getType().accept(this, arg);
		this.closeTag();
		this.openTag("Variables");
		for (VariableDeclarator var : n.getVariables()) {
			var.accept(this, arg);
		}
		this.closeTag();
		this.closeTag();
	}

	public void visit(ForeachStmt n, A arg) {
		this.openTag("ForeachStmt");
		this.openTag("Variable");
		n.getVariable().accept(this, arg);
		this.closeTag();
		this.openTag("Iterable");
		n.getIterable().accept(this, arg);
		this.closeTag();
		this.openTag("Body");
		n.getBody().accept(this, arg);
		this.closeTag();
		this.closeTag();
	}

	public void visit(ForStmt n, A arg) {
		this.openTag("ForStmt");
		if (n.getInit() != null) {
			this.openTag("Init");
			for (Expression e : n.getInit()) {
				e.accept(this, arg);
			}
			this.closeTag();
		}
		if (n.getCompare() != null) {
			this.openTag("Compare");
			n.getCompare().accept(this, arg);
			this.closeTag();
		}
		if (n.getUpdate() != null) {
			this.openTag("Update");
			for (Expression e : n.getUpdate()) {
				e.accept(this, arg);
			}
			this.closeTag();
		}
		this.openTag("Body");
		n.getBody().accept(this, arg);
		this.closeTag();
		this.closeTag();
	}

	public void visit(IfStmt n, A arg) {
		this.openTag("IfStmt");
		n.getCondition().accept(this, arg);
		this.closeTag();
		this.openTag("ThenStmt");
		n.getThenStmt().accept(this, arg);
		this.closeTag();
		if (n.getElseStmt() != null) {
			this.openTag("ElseStmt");
			n.getElseStmt().accept(this, arg);
			this.closeTag();            
		}
	}

	public void visit(ImportDeclaration n, A arg) {
		this.openTag("ImportDeclaration");
		n.getName().accept(this, arg);
		this.closeTag();
	}

	public void visit(InitializerDeclaration n, A arg) {
		this.openTag("InitializerDeclaration");
		if (n.getJavaDoc() != null) {
			n.getJavaDoc().accept(this, arg);
		}
		this.openTag("Block");
		n.getBlock().accept(this, arg);
		this.closeTag();
		this.closeTag();
	}

	public void visit(InstanceOfExpr n, A arg) {
		this.openTag("InstanceOfExpr");
		this.openTag("Expression");
		n.getExpr().accept(this, arg);
		this.closeTag();
		this.openTag("Type");
		n.getType().accept(this, arg);
		this.closeTag();
		this.closeTag();
	}

	public void visit(IntegerLiteralExpr n, A arg) {
		this.openTag("IntegerLiteralExpr");
		this.appendSource(n.getValue());
		this.closeTag();    	
	}

	public void visit(IntegerLiteralMinValueExpr n, A arg) {
		this.openTag("IntegerLiteralMinValueExpr");
		this.appendSource(n.getValue());
		this.closeTag();
	}

	public void visit(JavadocComment n, A arg) {
		this.openTag("JavaDocComment");
		this.appendSource(n.getContent());
		this.closeTag();
	}

	public void visit(LabeledStmt n, A arg) {
		this.openTag("LabeledStmt");
		n.getStmt().accept(this, arg);
		this.closeTag();
	}

	public void visit(LineComment n, A arg) {
	}

	public void visit(LongLiteralExpr n, A arg) {
		this.openTag("LongLiteralExpr");
		this.appendSource(n.getValue());
		this.closeTag(); 
	}

	public void visit(LongLiteralMinValueExpr n, A arg) {
		this.openTag("LongLiteralMinValueExpr");
		this.appendSource(n.getValue());
		this.closeTag();     	
	}

	public void visit(MarkerAnnotationExpr n, A arg) {
		this.openTag("MarkerAnnotationExpr");
		n.getName().accept(this, arg);
		this.closeTag();
	}

	public void visit(MemberValuePair n, A arg) {
		this.openTag("MemberValuePair");
		n.getValue().accept(this, arg);
		this.closeTag();
	}

	public void visit(MethodCallExpr n, A arg) {
		this.openTag("MethodCallExpr");
		this.openTag("Name");
		this.appendSource(n.getName());
		this.closeTag();      	
		if (n.getScope() != null) {
			this.openTag("Scope");
			n.getScope().accept(this, arg);
			this.closeTag();              
		}    
		if (n.getTypeArgs() != null) {
			this.openTag("TypeArguments");
			for (Type t : n.getTypeArgs()) {
				t.accept(this, arg);
			}
			this.closeTag();              
		}      
		if (n.getArgs() != null) {
			this.openTag("Arguments");
			for (Expression e : n.getArgs()) {
				e.accept(this, arg);
			}
			this.closeTag();            
		}
		this.closeTag();
	}

	public void visit(MethodDeclaration n, A arg) {
		this.openTag("MethodDeclaration");
		if (n.getJavaDoc() != null) {
			n.getJavaDoc().accept(this, arg);
		}
		if (n.getAnnotations() != null) {
			this.openTag("Annotations");
			for (AnnotationExpr a : n.getAnnotations()) {
				a.accept(this, arg);
			}
			this.closeTag();
		}
		if (n.getTypeParameters() != null) {
			this.openTag("TypeParameters");
			for (TypeParameter t : n.getTypeParameters()) {
				t.accept(this, arg);
			}
			this.closeTag();            
		}
		this.openTag("ReturnType");
		n.getType().accept(this, arg);
		this.closeTag();         
		if (n.getParameters() != null) {
			this.openTag("Parameters");
			for (Parameter p : n.getParameters()) {
				p.accept(this, arg);
			}
			this.closeTag();             
		}
		if (n.getThrows() != null) {
			this.openTag("Throws");
			for (NameExpr name : n.getThrows()) {
				name.accept(this, arg);
			}
			this.closeTag();             
		}
		if (n.getBody() != null) {
			this.openTag("Body");
			n.getBody().accept(this, arg);
			this.closeTag();            
		}
		this.closeTag();
	}

	public void visit(NameExpr n, A arg) {
		this.openTag("NameExpr");
		this.appendSource(n.getName());
		this.closeTag();    	
	}

	public void visit(NormalAnnotationExpr n, A arg) {
		n.getName().accept(this, arg);
		if (n.getPairs() != null) {
			for (MemberValuePair m : n.getPairs()) {
				m.accept(this, arg);
			}
		}
	}

	public void visit(NullLiteralExpr n, A arg) {
		this.openTag("NullLiteralExpr");
		this.appendSource(n.toString());
		this.closeTag();
	}

	public void visit(ObjectCreationExpr n, A arg) {
		this.openTag("ObjectCreationExpr");
		if (n.getScope() != null) {
			this.openTag("Scope");
			n.getScope().accept(this, arg);
			this.closeTag();
		}
		if (n.getTypeArgs() != null) {
			this.openTag("TypeArguments");
			for (Type t : n.getTypeArgs()) {
				t.accept(this, arg);
			}
			this.closeTag();
		}
		this.openTag("Type");
		n.getType().accept(this, arg);
		this.closeTag();
		if (n.getArgs() != null) {
			this.openTag("Arguments");
			for (Expression e : n.getArgs()) {
				e.accept(this, arg);
			}
			this.closeTag();
		}
		if (n.getAnonymousClassBody() != null) {
			this.openTag("AnonymousClassBody");
			for (BodyDeclaration member : n.getAnonymousClassBody()) {
				member.accept(this, arg);
			}
			this.closeTag();
		}
	}

	public void visit(PackageDeclaration n, A arg) {
		this.openTag("PackageDeclaration");
		if (n.getAnnotations() != null) {
			this.openTag("Annotations");
			for (AnnotationExpr a : n.getAnnotations()) {
				a.accept(this, arg);
			}
			this.closeTag();
		}
		this.openTag("Name");
		n.getName().accept(this, arg);
		this.closeTag();
		this.closeTag();
	}

	public void visit(Parameter n, A arg) {
		this.openTag("Parameter");
		if (n.getAnnotations() != null) {
			this.openTag("Annotations");
			for (AnnotationExpr a : n.getAnnotations()) {
				a.accept(this, arg);
			}
			this.closeTag();
		}
		this.openTag("Type");
		n.getType().accept(this, arg);
		this.closeTag();
		this.openTag("Id");
		n.getId().accept(this, arg);
		this.closeTag();
		this.closeTag();
	}

	public void visit(PrimitiveType n, A arg) {
		this.openTag("PrimitiveType");
		this.appendSource(n.toString());
		this.closeTag();
	}

	public void visit(QualifiedNameExpr n, A arg) {
		this.openTag("QualifiedNameExpr");
		this.openTag("Name");
		this.appendSource(n.getName());
		this.closeTag();
		this.openTag("Qualifier");
		n.getQualifier().accept(this, arg);
		this.closeTag();
		this.closeTag();
	}

	public void visit(ReferenceType n, A arg) {
		this.openTag("ReferenceType");
		n.getType().accept(this, arg);
		this.closeTag();
	}

	public void visit(ReturnStmt n, A arg) {
		if (n.getExpr() != null) {
			this.openTag("ReturnStmt");
			n.getExpr().accept(this, arg);
			this.closeTag();
		}
	}

	public void visit(SingleMemberAnnotationExpr n, A arg) {
		this.openTag("SingleMemberAnnotationExpr");
		this.openTag("Name");
		n.getName().accept(this, arg);
		this.closeTag();
		this.openTag("MemberValue");
		n.getMemberValue().accept(this, arg);
		this.closeTag();
		this.closeTag();         
	}

	public void visit(StringLiteralExpr n, A arg) {
		this.openTag("StringLiteralExpr");
		this.appendSource(n.getValue());
		this.closeTag();    	
	}

	public void visit(SuperExpr n, A arg) {
		if (n.getClassExpr() != null) {
			this.openTag("SuperExpr");
			n.getClassExpr().accept(this, arg);
			this.closeTag();
		}
	}

	public void visit(SwitchEntryStmt n, A arg) {
		this.openTag("SwitchEntryStmt");
		if (n.getLabel() != null) {
			this.openTag("Label");
			n.getLabel().accept(this, arg);
			this.closeTag();
		}
		if (n.getStmts() != null) {
			this.openTag("Statements");
			for (Statement s : n.getStmts()) {
				s.accept(this, arg);
			}
			this.closeTag();
		}
		this.closeTag();
	}

	public void visit(SwitchStmt n, A arg) {
		this.openTag("SwitchStmt");
		this.openTag("Selector");
		n.getSelector().accept(this, arg);
		this.closeTag();
		if (n.getEntries() != null) {
			this.openTag("Entries");
			for (SwitchEntryStmt e : n.getEntries()) {
				e.accept(this, arg);
			}
			this.closeTag();
		}
		this.closeTag();
	}

	public void visit(SynchronizedStmt n, A arg) {
		this.openTag("SynchronizedStmt");
		this.openTag("Expression");
		n.getExpr().accept(this, arg);
		this.closeTag();
		this.openTag("Block");
		n.getBlock().accept(this, arg);
		this.closeTag();
		this.closeTag();
	}

	public void visit(ThisExpr n, A arg) {
		this.openTag("ThisExpr");
		if (n.getClassExpr() != null) {
			n.getClassExpr().accept(this, arg);
		}
		this.closeTag();
	}

	public void visit(ThrowStmt n, A arg) {
		this.openTag("ThrowStmt");
		n.getExpr().accept(this, arg);
		this.closeTag();
	}

	public void visit(TryStmt n, A arg) {
		this.openTag("TryStmt");
		this.openTag("TryBlock");
		n.getTryBlock().accept(this, arg);
		this.closeTag();
		if (n.getCatchs() != null) {
			this.openTag("Catchs");
			for (CatchClause c : n.getCatchs()) {
				c.accept(this, arg);
			}
			this.closeTag();
		}
		if (n.getFinallyBlock() != null) {
			this.openTag("FinallyBlock");
			n.getFinallyBlock().accept(this, arg);
			this.closeTag();
		}
		this.closeTag();
	}

	public void visit(TypeDeclarationStmt n, A arg) {
		this.openTag("TypeDeclarationStmt");
		n.getTypeDeclaration().accept(this, arg);
		this.closeTag();
	}

	public void visit(TypeParameter n, A arg) {
		this.openTag("TypeParameter");
		this.openTag("Name");
		this.appendSource(n.getName());
		this.closeTag();
		if (n.getTypeBound() != null) {
			this.openTag("TypeBound");
			for (ClassOrInterfaceType c : n.getTypeBound()) {
				c.accept(this, arg);
			}
			this.closeTag();
		}
		this.closeTag();
	}

	public void visit(UnaryExpr n, A arg) {
		this.openTag("UnaryExpr");
		this.openTag("Operator");
		this.appendSource(n.getOperator().name());
		this.closeTag();
		this.openTag("Expression");
		n.getExpr().accept(this, arg);
		this.closeTag();
		this.closeTag();
	}

	public void visit(VariableDeclarationExpr n, A arg) {
		this.openTag("VariableDeclarationExpr");
		if (n.getAnnotations() != null) {
			this.openTag("Annotations");
			for (AnnotationExpr a : n.getAnnotations()) {
				a.accept(this, arg);
			}
			this.closeTag();
		}
		this.openTag("Type");
		n.getType().accept(this, arg);
		this.closeTag();
		this.openTag("Variables");
		for (VariableDeclarator v : n.getVars()) {
			v.accept(this, arg);
		}
		this.closeTag();
		this.closeTag();
	}

	public void visit(VariableDeclarator n, A arg) {
		this.openTag("VariableDeclarator");
		this.openTag("Id");
		n.getId().accept(this, arg);
		this.closeTag();
		if (n.getInit() != null) {
			this.openTag("Init");
			n.getInit().accept(this, arg);
			this.closeTag();
		}
		this.closeTag();
	}

	public void visit(VariableDeclaratorId n, A arg) {
		this.openTag("VariableDeclaratorId");   
		this.appendSource(n.getName());
		this.closeTag();    	
	}

	public void visit(VoidType n, A arg) {
		this.openTag("VoidType");
		this.closeTag();
	}

	public void visit(WhileStmt n, A arg) {
		this.openTag("WhileStmt");
		this.openTag("Condition");
		n.getCondition().accept(this, arg);
		this.closeTag();     
		this.openTag("Body");
		n.getBody().accept(this, arg);
		this.closeTag(); 
		this.closeTag();            
	}

	public void visit(WildcardType n, A arg) {
		this.openTag("WildcardType");
		if (n.getExtends() != null) {
			this.openTag("Extends");
			n.getExtends().accept(this, arg);
			this.closeTag();
		}
		if (n.getSuper() != null) {
			this.openTag("Super");
			n.getSuper().accept(this, arg);
			this.closeTag();
		}
		this.closeTag();
	}
}